<?php

namespace Codemanas\ZoomWooCommerceAddon;

use Codemanas\ZoomWooCommerceAddon\Admin\AdminNoticeHandler;

/**
 * Class Bootstrap
 *
 * Bootstrap our plugin
 *
 * @author  Deepen Bajracharya, CodeManas, 2020. All Rights reserved.
 * @since   1.0.0
 * @package Codemanas\ZoomWooCommerceAddon
 */
class Bootstrap {
	private $min_required_php_version = '7.4.0';
	private $min_required_zoom_plugin_version = '3.6.0';
	private $plugin_name = VZAPI_WOOCOMMERCE_ADDON_PLUGIN;
	public static $item_id = 1986;
	public static $store_url = 'https://www.codemanas.com';
	public static $options_page = 'edit.php?post_type=zoom-meetings&page=woocommerce&tab=license';
	private static $license = '_vczapi_woocommerce_addon_license';

	private static $_instance = null;

	private $key_validate;

	private $plugin_settings;

	/**
	 * Create only one instance so that it may not Repeat
	 *
	 * @since 1.0.0
	 */
	public static function get_instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Bootstrap constructor.
	 */
	public function __construct() {
		//need to load autoloader before we call the AdminNoticeHandler
		require_once VZAPI_WOOCOMMERCE_ADDON_DIR_PATH . 'vendor/autoload.php';
		
		//activation hooks
		register_activation_hook( VZAPI_WOOCOMMERCE_ADDON_FILE_PATH, 'Codemanas\ZoomWooCommerceAddon\Bootstrap::activate' );
		register_activation_hook( VZAPI_WOOCOMMERCE_ADDON_FILE_PATH, 'Codemanas\ZoomWooCommerceAddon\CronHandlers::activate_cron' );
		if ( ! Helper::check_is_booking_addon_active() ) {
			//don't deactivate if booking  addon is still active
			register_deactivation_hook( VZAPI_WOOCOMMERCE_ADDON_FILE_PATH, 'Codemanas\ZoomWooCommerceAddon\CronHandlers::deactivate_cron' );
		}

		//load text domain
		$this->load_text_domain();

		//get the license details
		$this->key_validate    = trim( get_option( self::$license ) );
		$this->plugin_settings = get_option( '_vczapi_woocommerce_settings' );

		//Set up a listener to show and display admin messages.
		AdminNoticeHandler::get_instance();
		add_action( 'plugins_loaded', [ $this, 'load_modules' ] );
	}

	public function load_text_domain() {
		load_plugin_textdomain( 'vczapi-woocommerce-addon', false, plugin_basename( dirname( __FILE__ ) ) . '/languages/' );
	}

	public function load_modules() {
		//check if all required dependencies are met
		if ( ! $this->check_dependencies() ) {
			return;
		}

		//load all modules
		$this->load();

		//these modules are only needed in backend
		add_action( 'admin_init', array( $this, 'updater' ), 1 );
		add_action( 'admin_init', array( $this, 'load_admin' ), 1 );

		//enqueue the scripts
		add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ) );
	}

	public static function activate() {
		//Check if WooCommerce Exists
		if ( ! is_plugin_active( 'video-conferencing-with-zoom-api/video-conferencing-with-zoom-api.php' ) and current_user_can( 'activate_plugins' ) ) {
			$exit_msg = __( "This Plugin requires Video Conferencing with zoom api plugin to work. Please install it first to use this.", "vczapi-woocommerce-addon" );
			wp_die( $exit_msg );
		}

		//Create rewrite endpoint
		add_rewrite_endpoint( 'wc-zoom-meetings', EP_ROOT | EP_PAGES );
		add_rewrite_endpoint( 'wc-zoom-recordings', EP_ROOT | EP_PAGES );
		flush_rewrite_rules();
	}

	/**
	 * Load Admin Classes
	 */
	public function load_admin() {
		if ( ! empty( $this->key_validate ) ) {
			//Normal WooCommerce Product
			$this->autowire( Admin\WooCommerceTable::class );
			$this->autowire( Admin\ZoomMetaBox::class );
		}
	}

	/**
	 * Load Dependencies
	 */
	public function load() {
		if ( ! empty( $this->key_validate ) ) {
			$this->autowire( TemplateOverrides::class );
			$this->autowire( Admin\ProductType::class );
			Admin\WooCommerceZoomConnection::get_instance();
			Orders::get_instance();

			if ( vczapi_wc_product_vendors_addon_active() ) {
				ProductVendors::get_instance();
			}

			$this->autowire( CronHandlers::class );
			Cart::get_instance();
			$this->autowire( Shortcode::class );

			if ( class_exists( 'SitePress' ) ) {
				WPML::get_instance();
			}
		}

		$this->autowire( Admin\Activator::class );
		$this->autowire( Admin\Settings::class );
	}

	/**
	 * Dependency Injection Process
	 *
	 * @param $obj
	 *
	 * @since  1.0.2
	 * @author Deepen Bajracharya
	 */
	private function autowire( $obj ) {
		new $obj;
	}

	/**
	 * Updater
	 */
	public static function updater() {
		$license_key = trim( get_option( self::$license ) );
		new Updater( self::$store_url, VZAPI_WOOCOMMERCE_ADDON_DIR_PATH . 'vczapi-woocommerce-addon.php', array(
			'version' => VZAPI_WOOCOMMERCE_ADDON_PLUGIN_VERSION,
			'license' => $license_key,
			'author'  => 'Codemanas',
			'item_id' => self::$item_id,
			'beta'    => false,
		) );

		#$updater->check();
	}

	/**
	 * Enqueue Scripts in frontend side
	 */
	public function scripts() {
		wp_register_script( 'vczapi-woocommerce-script', VZAPI_WOOCOMMERCE_ADDON_DIR_URI . 'assets/frontend/js/scripts.min.js', array( 'jquery' ), VZAPI_WOOCOMMERCE_ADDON_PLUGIN_VERSION, true );
		wp_enqueue_style( 'vczapi-woocommerce-style', VZAPI_WOOCOMMERCE_ADDON_DIR_URI . 'assets/frontend/css/style.min.css', false, VZAPI_WOOCOMMERCE_ADDON_PLUGIN_VERSION );
	}

	private function check_dependencies() {
		$active_plugins = apply_filters( 'active_plugins', get_option( 'active_plugins' ) );
		//check required php version
		if ( version_compare( phpversion(), $this->min_required_php_version, '<' ) ) {
			AdminNoticeHandler::set_message(
				sprintf( '<strong>%s</strong> requires PHP Version %s or higher. Please update the PHP Version of your system',
					$this->plugin_name,
					$this->required_php_version ),
				'error'
			);

			return false;
		}


		if ( ! defined( 'ZVC_PLUGIN_VERSION' ) && ! in_array( 'video-conferencing-with-zoom-api/video-conferencing-with-zoom-api.php', $active_plugins ) ) {
			AdminNoticeHandler::set_message( sprintf( '%s requires <a href="https://wordpress.org/plugins/video-conferencing-with-zoom-api/">Video Conferencing with Zoom API</a>', $this->plugin_name ), 'error' );

			return false;
		}

		//check if ZVC_PLUGIN_VERSION is defined if not see if it's an older option via get_plugin_version - Basically we are checking if the required version of Video Conferencing with Zoom is loaded.
		$zoom_core_version = defined( 'ZVC_PLUGIN_VERSION' ) ? ZVC_PLUGIN_VERSION : $this->get_plugin_version( 'video-conferencing-with-zoom-api/video-conferencing-with-zoom-api.php' );
		if ( defined( 'ZVC_PLUGIN_VERSION' ) && version_compare( $zoom_core_version, $this->min_required_zoom_plugin_version, '<' ) ) {
			AdminNoticeHandler::set_message( sprintf( '<a href="https://wordpress.org/plugins/video-conferencing-with-zoom-api/" target="_blank" referrerpolicy="no-referrer">Video Conferencing with Zoom</a> version
            <strong>%s</strong> or greater is required for <strong>%s</strong> to function properly.
Please update the plugin <a href="%s">from plugins page</a> to version <strong>%s</strong> or
	greater.',
				$this->min_required_zoom_plugin_version,
				$this->plugin_name,
				admin_url( '/plugins.php' ),
				$this->min_required_zoom_plugin_version ),
				'error' );

			return false;
		}


		//load woocommerce checkers
		require_once VZAPI_WOOCOMMERCE_ADDON_DIR_PATH . 'woo-dependencies.php';
		if ( ! vczapi_woocommerce_addon_active() ) {
			AdminNoticeHandler::set_message( sprintf( '<strong>%s requires.</strong> <a href="http://wordpress.org/extend/plugins/woocommerce/">WooCommerce</a> must to be installed and active to work. Please install or activate WooCommerce first.</p>',
				VZAPI_WOOCOMMERCE_ADDON_PLUGIN ),
				'error' );
		}

		return true;

	}

	/**
	 * Required just in case ZVC_PLUGIN_VERSION was not defined for some reason
	 *
	 * @param string $plugin
	 *
	 */
	private function get_plugin_version( string $plugin ) {
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		}
		$active_plugins = apply_filters( 'active_plugins', get_option( 'active_plugins' ) );
		if ( ! in_array( $plugin, $active_plugins ) ) {
			return '0.0.1';
		}
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );

		return ! empty( $plugin_data['Version'] ) ? $plugin_data['Version'] : '0.0.0';
	}
}

Bootstrap::get_instance();